
/*  ************************************************************************  *
 *                                  main.cpp                                  *
 *  ************************************************************************  */

#include    "stdinc.h"

#include    <stdlib.h>          // for errno and wcstoul

#include    "profile.h"
#include    "puterror.h"
#include    "test.h"

/*  ************************************************************************  */

static CHAR const Description [] =
"Tests Windows profiling\n"
"\n"
"Profile options:\n"
"\n"
"  /bucketsize:n     specify bucket size (default is 4)\n"
"  /interval:n       specify interval (or 0 to use source's default)\n"
"  /source:n         specify profile source (default is 0 for timer)\n"
"\n"
"Test options:\n"
"\n"
"  /runs:n           number of times to run test (default is 10000)\n"
"\n"
"Placeholders:\n"
"\n"
"  n                 integer: use 0x prefix for hexadecimal"
"\n"
"Specifying a profile interval may require administrative elevation.";

/*  ************************************************************************  */
/*  Forward references  */

static bool ParseParameterColonValueNumeric (PWSTR, PCWSTR, ULONG *);

/*  ========================================================================  */
/*  The actual program  */

int __cdecl wmain (int argc, PWSTR *argv)
{
    /*  Profile options, with defaults  */

    ULONG bucketsize = sizeof (ULONG);
    ULONG interval = 0;
    KPROFILE_SOURCE source = ProfileTime;

    /*  Test options, with defaults  */

    ULONG runs = 10000;

    /*  Command-line parsing  */

    if (argc == 0) return -1;

    while (++ argv, -- argc != 0) {
        PWSTR arg = *argv;
        PWSTR p = arg;
        if (*p == L'-' OR *p == L'/') {
            p ++;

            if (_wcsicmp (p, L"?") == 0 OR _wcsicmp (p, L"help") == 0) {
                PutInfo (Description);
                return 1;
            }

            ULONG ul;
            if (ParseParameterColonValueNumeric (p, L"bucketsize", &ul)) {
                bucketsize = ul;
                continue;
            }
            else if (ParseParameterColonValueNumeric (p, L"interval", &ul)) {
                interval = ul;
                continue;
            }
            else if (ParseParameterColonValueNumeric (p, L"source", &ul)) {
                source = (KPROFILE_SOURCE) ul;
                continue;
            }
            else if (ParseParameterColonValueNumeric (p, L"runs", &ul)) {
                runs = ul;
                continue;
            }

            PutError ("Invalid switch \"%ws\"", arg);
            return -1;
        }

        PutError ("Too many parameters \"%ws\"", arg);
        return -1;
    }

    /*  Create and initialise a CProfile. This helper class sets up the
        profiling of all code that's put in a pre-defined area of the
        program.  */

    CProfile profile;
    ULONG numbuckets;
    DWORD ec = profile.Init (bucketsize, interval, source, &numbuckets);
    if (ec == ERROR_SUCCESS) {

        /*  What we have in that pre-defined area is a test routine that
            runs differently depending on whether the data it works with is
            sorted in advance. The idea is to model an investigation into
            how real-world code gets different performance when run with
            different parameters. Here, we take two profiles, first with
            the test routine working on unsorted data, then with sorted
            data.  */

        InitTest ();

        ULONG *counts [2];
        ULONG results [RTL_NUMBER_OF (counts)];
        ULONG times [RTL_NUMBER_OF (counts)];
        ULONG n;
        for (n = 0; n < RTL_NUMBER_OF (counts); n ++) {

            if (n != 0) ChangeTest ();

            /*  Start (or restart) the profiling. Run the test however many
                times are thought, e.g., from the command line, might yield
                a good sample. Then stop profiling and collect a copy of the
                execution counts to show later.

                As elaborations, report how long the multiple runs have
                taken - and also provide that the test returns a result so
                that we can verify that the change we made to what it works
                with was consequential only for performance.

                Warning 28159, which might otherwise be flagged by
                Microsoft's tools for static code analysis, is disabled here
                for the simplicity of allowing that the program can easily
                be built for Windows versions that predate Windows Vista and
                its GetTickCount64 function. We anyway get the tick count
                only to compute differences. Given that our particular test
                makes no external calls, it's vanishingly improbable that we
                could lose 49 days - and even if we did, what would it
                matter?  */

            ec = profile.Start ();
            if (ec != ERROR_SUCCESS) break;

            #pragma warning (push)
            #pragma warning (disable : 28159)

            times [n] = GetTickCount ();

            ULONG volatile result = 0;
            for (ULONG i = runs; i != 0; i --) {
                result = RunTest ();
            }
            results [n] = result;

            times [n] = GetTickCount () - times [n];

            #pragma warning (pop)   // disable : 28159

            ec = profile.Stop (&counts [n]);
            if (ec != ERROR_SUCCESS) break;
        }

        if (n >= 2) {

            /*  Our particular test should have given the same result both
                times - but confirm this!  */

            if (results [0] != results [1]) {

                PutError (
                    "Unsorted and sorted results differ: 0x%08X and 0x%08X",
                    results [0],
                    results [1]);
            }

            /*  Show the execution times side by side.  */

            PutInfo (
                "Times for %u runs are %u.%03u and %u.%03u seconds",
                runs,
                times [0] / 1000,
                times [0] % 1000,
                times [1] / 1000,
                times [1] % 1000);

            /*  Show the two sets of execution counts side by side, with
                totals to come at the end.  */

            PutInfo ("Offset        Unsorted     Sorted");

            ULONG total0 = 0;
            ULONG total1 = 0;
            bool overflow = false;

            ULONG offset;
            ULONG *p0;
            ULONG *p1;
            ULONG n;
            for (offset = 0, p0 = counts [0], p1 = counts [1], n = numbuckets;
                    n != 0;
                    offset += bucketsize, p0 ++, p1 ++, n --) {

                if (NOT overflow) {
                    total0 += *p0;
                    if (total0 < *p0) overflow = true;
                    total1 += *p1;
                    if (total1 < *p1) overflow = true;
                }

                /*  Let's present nicely for users who don't think in
                    hexadecimal.  */

                PutInfo ("0x%08X: %10u %10u", offset, *p0, *p1);
            }

            if (NOT overflow) {
                PutInfo ("            ========== ==========", total0, total1);
                PutInfo ("            %10u %10u", total0, total1);
            }
        }

        /*  It's all clean-up from here.  */

        while (n -- != 0) {
            delete counts [n];
        }
    }
    return (int) ec;
}

/*  ************************************************************************  */
/*  Helpers  */

bool ParseParameterColonValue (PWSTR Argument, PCWSTR Parameter, PWSTR *Value)
{
    PWSTR p = Argument;
    size_t cchparm = wcslen (Parameter);
    if (_wcsnicmp (p, Parameter, cchparm) == 0) {
        p += cchparm;
        if (*p == L':') {
            p ++;
            if (*p != L'\0') {
                *Value = p;
                return true;
            }
        }
    }
    return false;
}

bool ParseValueNumeric (PCWSTR Value, ULONG *Data)
{
    PWCHAR end;
    ULONG n = wcstoul (Value, &end, 0);
    if ((n != 0 OR end != Value) AND *end == L'\0' AND errno == 0) {
        *Data = n;
        return true;
    }
    return false;
}

bool
ParseParameterColonValueNumeric (
    PWSTR Argument,
    PCWSTR Parameter,
    ULONG *Data)
{
    PWSTR value;
    if (ParseParameterColonValue (Argument, Parameter, &value)) {
        if (ParseValueNumeric (value, Data)) return true;
    }
    return false;
}

/*  ************************************************************************  */

